import os
import json
import torch
import argparse
from tqdm import tqdm
from datasets import load_dataset
from transformers import AutoTokenizer
import re
import sys
import random
import numpy as np


torch.manual_seed(42)
torch.cuda.manual_seed(42)
torch.cuda.manual_seed_all(42)
np.random.seed(42)
random.seed(42)
torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmark = False

sys.path.append("/fs/nexus-scratch/hjae/ShadowKV")

from models.llama import LlamaForCausalLM  

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--model_path", type=str, default="meta-llama/Meta-Llama-3.1-8B-Instruct")
    parser.add_argument("--token_budget", type=int, default=1024)
    parser.add_argument("--compression_enabled", action="store_true")
    parser.add_argument("--compression_threshold", type=int, default=128)
    parser.add_argument("--compression_ratio", type=float, default=0.5)
    parser.add_argument("--window_size", type=int, default=512)  
    parser.add_argument("--max_samples", type=int, default=100)
    parser.add_argument("--output_dir", type=str, default="results")
    return parser.parse_args()

def load_model_and_tokenizer(args):
    print(f"Loading model from {args.model_path}...")
    tokenizer = AutoTokenizer.from_pretrained(args.model_path)

    model = LlamaForCausalLM.from_pretrained(
        args.model_path,
        device_map="auto",
        torch_dtype=torch.float16
    )

    model.shadowkv_init( 
        window_size=args.window_size,
        max_tokens=args.token_budget,
        compress_ratio=args.compression_ratio if args.compression_enabled else 1.0,
        compress_threshold=args.compression_threshold,
    )

    return model, tokenizer

def format_prompt(context, query, options):
    options_str = "\n".join([f"{chr(65+i)}: {opt}" for i, opt in enumerate(options)])
    return f"""Read the following passage and answer the question by selecting the correct option (A, B, C, or D).\n\nPassage: {context}\n\nQuestion: {query}\n\nOptions:\n{options_str}\n\nAnswer:"""

def extract_answer(response):
    match = re.search(r"([A-D])", response)
    if match:
        return match.group(1)
    return None

def is_correct(pred, correct_idx):
    if pred is None:
        return False
    return pred == chr(65 + correct_idx)  # 0->A, 1->B, 2->C, 3->D

def load_partial_results(output_file):
    if os.path.exists(output_file):
        with open(output_file, "r") as f:
            data = json.load(f)
            return data.get("results", []), data.get("correct", 0), data.get("total", 0)
    return [], 0, 0

def evaluate_logiqa(model, tokenizer, dataset, args, output_file):
    results, correct, total = load_partial_results(output_file)
    start_idx = total
    n_total = min(len(dataset), args.max_samples) if args.max_samples else len(dataset)
    batch_size = 100
    save_every = 50

    while start_idx < n_total:
        end_idx = min(start_idx + batch_size, n_total)
        subset = dataset.select(range(start_idx, end_idx))
        for sample in tqdm(subset, initial=start_idx, total=n_total):
            context = sample["context"]
            query = sample["query"]
            options = sample["options"]
            correct_idx = sample["correct_option"]

            prompt = format_prompt(context, query, options)
            inputs = tokenizer(prompt, return_tensors="pt").to(model.device)

            with torch.no_grad():
                outputs = model.generate(
                    inputs.input_ids,
                    max_new_tokens=128,  
                    temperature=0.7,
                    top_p=0.9,
                    use_cache=True
                )

            response = tokenizer.decode(outputs[0], skip_special_tokens=True)
            predicted_answer = extract_answer(response)

            correct_flag = is_correct(predicted_answer, correct_idx)
            if correct_flag:
                correct += 1
            total += 1

            results.append({
                "context": context,
                "query": query,
                "options": options,
                "correct_option": chr(65 + correct_idx),
                "predicted_answer": predicted_answer,
                "is_correct": correct_flag,
                "response": response
            })

            if (total % save_every == 0) or (total == n_total):
                with open(output_file, "w") as f:
                    json.dump({
                        "args": vars(args),
                        "accuracy": correct / total if total > 0 else 0,
                        "results": results,
                        "correct": correct,
                        "total": total
                    }, f, indent=2)
                print(f"[Checkpoint] Saved {total} results to {output_file}")
        start_idx = end_idx
    return results, correct / total if total > 0 else 0

def main():
    args = parse_args()
    os.makedirs(args.output_dir, exist_ok=True)

    model, tokenizer = load_model_and_tokenizer(args)

    print("Loading LogiQA dataset...")
    dataset = load_dataset("lucasmccabe/logiqa")
    test_set = dataset["validation"]
    

    if args.max_samples is not None:
        test_set = test_set.shuffle(seed=42).select(range(min(args.max_samples, len(test_set))))
        print(f"Using {len(test_set)} samples for evaluation (randomly selected with seed=42)")

    print("Starting evaluation...")
    output_file = os.path.join(args.output_dir, "logiqa_results_shadowkv.json")
    results, accuracy = evaluate_logiqa(model, tokenizer, test_set, args, output_file)

    print(f"\nFinal accuracy: {accuracy:.2%}")
    print(f"Results saved to {output_file}")

if __name__ == "__main__":
    main()
